import LoadingCircle from "@/components/icons/loading.circle";
import { Button } from "@/components/ui/button";
import { DialogClose } from "@/components/ui/dialog";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { InputTextarea } from "@/components/ui/input.textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { errorToStringFormatter } from "@/lib/formatters/error-formatters";
import { TransformedAgencies } from "@/types/transformers";
import { TODO } from "@/types/utils";
import { Asterisk } from "lucide-react";
import { SubmitHandler, UseFormReturn } from "react-hook-form";
import { AppFormErrors } from "../type/errors";
import { AppStatewideServiceFormData } from "../zod/schema.form";

const INTERNAL__CANCEL_TEXT = 'Cancel'
const INTERNAL__SUBMIT_TEXT = 'Submit'

export function WorkflowFormStatewideServices({
  form,
  handleSubmit,
  agencies,
  errors,
  overrides,
}: {
  form: UseFormReturn<AppStatewideServiceFormData, TODO, undefined>,
  handleSubmit: SubmitHandler<AppStatewideServiceFormData>,
  agencies: TransformedAgencies,
  errors: AppFormErrors,
  overrides?: {
    cancelText?: string,
    submitText?: string,
  }
}) {
  const { formState: { isSubmitting } } = form
  const cancelText = overrides?.cancelText ?? INTERNAL__CANCEL_TEXT
  const submitText = overrides?.submitText ?? INTERNAL__SUBMIT_TEXT

  return (
    <Form {...form}>
      <form
        onSubmit={form.handleSubmit(handleSubmit)}
        className="space-y-4"
      >
        <FormField
          control={form.control}
          name="service_agency"
          render={({ field: { value, onChange, ...field } }) => (
            <FormItem>
              <FormLabel className="flex flex-row gap-0.5">Agency identifier <Asterisk className="text-destructive size-3" /></FormLabel>
              <FormControl>
                <Select
                  onValueChange={(value) => onChange(value)}
                  defaultValue={value}
                >
                  <SelectTrigger>
                    <SelectValue
                      placeholder="Select associated agency"
                      {...field}
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {agencies.map((a, i) => {
                      return (
                        <SelectItem key={a.gtfsId ?? i} value={a.gtfsId!}>{a.gtfsId}{" - "}{a.name}</SelectItem>
                      )
                    })}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="service_alias"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="flex flex-row gap-0.5">Service name <Asterisk className="text-destructive size-3" /></FormLabel>
              <FormControl>
                <Input
                  placeholder="acme"
                  type="text"
                  {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="service_url"
          render={({ field: { onChange, ...field } }) => (
            <FormItem>
              <FormLabel>Website</FormLabel>
              <FormControl>
                <Input
                  placeholder="https://www.•••••••••.com"
                  type="url"
                  onChange={(event) => event.currentTarget.value === "" ? onChange(undefined) : onChange(event)}
                  {...field} />
              </FormControl>
              <FormDescription>Url of the site users can visit.</FormDescription>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="service_phone"
          render={({ field: { onChange, ...field } }) => (
            <FormItem>
              <FormLabel>Phone number</FormLabel>
              <FormControl>
                <Input
                  placeholder="+1 ••• ••• ••••"
                  type="phone"
                  onChange={(event) => event.currentTarget.value === "" ? onChange(undefined) : onChange(event)}
                  {...field}
                />
              </FormControl>
              <FormDescription>Phone number of the line users can call.</FormDescription>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="service_message"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="flex flex-row gap-0.5">Message <Asterisk className="text-destructive size-3" /></FormLabel>
              <FormControl>
                <InputTextarea
                  placeholder="Default message to show users when shown service."
                  {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        {
          !!errors && (
            errors.map((e, i) => {
              return (
                <div key={i} className="text-destructive text-xs" > {errorToStringFormatter(e)} </div>
              )
            })
          )
        }

        <div className="flex flex-row gap-2 justify-end">
          <DialogClose asChild>
            <Button variant={"secondary"}>{cancelText}</Button>
          </DialogClose>

          <Button type="submit" disabled={isSubmitting}>
            {submitText}{" "}
            {isSubmitting && <LoadingCircle className="ml-2" />}
          </Button>
        </div>
      </form>
    </Form>

  )
}
